/*
 * Decompiled with CFR 0.152.
 */
package device.rtl;

import common.Config;
import common.Log;
import decoder.Broadcaster;
import decoder.ComplexBuffer;
import decoder.FloatAveragingBuffer;
import decoder.Listener;
import decoder.SourceUSB;
import device.ByteSampleAdapter;
import device.DeviceException;
import device.DevicePanel;
import device.ThreadPoolManager;
import device.TunerController;
import device.TunerType;
import device.rtl.RTLPanel;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import javax.usb.UsbDisconnectedException;
import javax.usb.UsbException;
import org.usb4java.Device;
import org.usb4java.DeviceDescriptor;
import org.usb4java.DeviceHandle;
import org.usb4java.LibUsb;
import org.usb4java.LibUsbException;
import org.usb4java.Transfer;
import org.usb4java.TransferCallback;

public abstract class RTL2832TunerController
extends TunerController {
    public static final int INT_NULL_VALUE = -1;
    public static final long LONG_NULL_VALUE = -1L;
    public static final double DOUBLE_NULL_VALUE = -1.0;
    public static final int TWO_TO_22_POWER = 0x400000;
    public static final byte USB_INTERFACE = 0;
    public static final byte USB_BULK_ENDPOINT = -127;
    public static final byte CONTROL_ENDPOINT_IN = -64;
    public static final byte CONTROL_ENDPOINT_OUT = 64;
    public static final long TIMEOUT_US = 1000000L;
    public static final byte REQUEST_ZERO = 0;
    public static final int TRANSFER_BUFFER_POOL_SIZE = 16;
    public static final byte EEPROM_ADDRESS = -96;
    public static final byte[] sFIR_COEFFICIENTS = new byte[]{-54, -36, -41, -40, -32, -14, 14, 53, 6, 80, -100, 13, 113, 17, 20, 113, 116, 25, 65, -91};
    public static final SampleRate DEFAULT_SAMPLE_RATE = SampleRate.RATE_0_240MHZ;
    protected Device mDevice;
    protected DeviceDescriptor mDeviceDescriptor;
    protected DeviceHandle mDeviceHandle;
    private SampleRate mSampleRate = DEFAULT_SAMPLE_RATE;
    private BufferProcessor mBufferProcessor;
    private ByteSampleAdapter mSampleAdapter = new ByteSampleAdapter();
    private Broadcaster<ComplexBuffer> mComplexBufferBroadcaster = new Broadcaster();
    private LinkedTransferQueue<byte[]> mFilledBuffers = new LinkedTransferQueue();
    private SampleRateMonitor mSampleRateMonitor;
    private AtomicInteger mSampleCounter = new AtomicInteger();
    private static final DecimalFormat mDecimalFormatter = new DecimalFormat("###,###,###.0");
    private static final DecimalFormat mPercentFormatter = new DecimalFormat("###.00");
    protected int mOscillatorFrequency = 28800000;
    public int mBufferSize = 131072;
    protected Descriptor mDescriptor;
    private ThreadPoolManager mThreadPoolManager;
    SourceUSB usbSource;
    boolean devicePulled;

    public RTL2832TunerController(Device device, DeviceDescriptor deviceDescriptor, ThreadPoolManager threadPoolManager, long minTunableFrequency, long maxTunableFrequency, int centerUnusableBandwidth, double usableBandwidthPercentage) throws DeviceException {
        super("RTL", minTunableFrequency, maxTunableFrequency);
        this.mThreadPoolManager = threadPoolManager;
        this.mDevice = device;
        this.mDeviceDescriptor = deviceDescriptor;
    }

    public void init(SampleRate sampleRate) throws DeviceException {
        this.mDeviceHandle = new DeviceHandle();
        int result = LibUsb.open(this.mDevice, this.mDeviceHandle);
        if (result != 0) {
            this.mDeviceHandle = null;
            throw new DeviceException("libusb couldn't open RTL2832 usb device [" + LibUsb.errorName(result) + "]");
        }
        RTL2832TunerController.claimInterface(this.mDeviceHandle);
        try {
            this.setSampleRate(sampleRate);
        }
        catch (Exception e) {
            throw new DeviceException("RTL2832 Tuner Controller - couldn't set default sample rate " + e.getMessage());
        }
        byte[] eeprom = null;
        try {
            eeprom = this.readEEPROM(this.mDeviceHandle, (short)0, 256);
        }
        catch (Exception e) {
            Log.errorDialog("error while reading the EEPROM device descriptor", e.getMessage());
        }
        try {
            this.mDescriptor = new Descriptor(eeprom);
            if (eeprom == null) {
                Log.errorDialog("eeprom byte array was null - constructed ", "empty descriptor object");
            }
        }
        catch (Exception e) {
            Log.errorDialog("error while constructing device descriptor using descriptor byte array " + (eeprom == null ? "[null]" : Arrays.toString(eeprom)), e.getMessage());
        }
        this.name = "RTL R820T";
    }

    public static void claimInterface(DeviceHandle handle) throws DeviceException {
        if (handle != null) {
            int result = LibUsb.kernelDriverActive(handle, 0);
            if (result == 1 && (result = LibUsb.detachKernelDriver(handle, 0)) != 0) {
                Log.errorDialog("ERROR", "failed attempt to detach kernel driver [" + LibUsb.errorName(result) + "]");
                throw new DeviceException("couldn't detach kernel driver from device");
            }
            result = LibUsb.claimInterface(handle, 0);
            if (result != 0) {
                throw new DeviceException("couldn't claim usb interface [" + LibUsb.errorName(result) + "]");
            }
        } else {
            throw new DeviceException("couldn't claim usb interface - no device handle");
        }
    }

    public static void releaseInterface(DeviceHandle handle) throws DeviceException {
        int result = LibUsb.releaseInterface(handle, 0);
        if (result != 0) {
            throw new DeviceException("couldn't release interface [" + LibUsb.errorName(result) + "]");
        }
    }

    public Descriptor getDescriptor() {
        if (this.mDescriptor != null && this.mDescriptor.isValid()) {
            return this.mDescriptor;
        }
        return null;
    }

    public void setSamplingMode(SampleMode mode) throws LibUsbException {
        switch (mode) {
            case QUADRATURE: {
                this.setIFFrequency(0);
                RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ZERO, (short)8, 205, 1);
                RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)177, 27, 1);
                RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ZERO, (short)6, 128, 1);
                break;
            }
            default: {
                throw new LibUsbException("QUADRATURE mode is the only mode currently supported", -12);
            }
        }
    }

    public void setIFFrequency(int frequency) throws LibUsbException {
        long ifFrequency = 0x400000L * (long)frequency / (long)this.mOscillatorFrequency * -1L;
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)25, (short)(Long.rotateRight(ifFrequency, 16) & 0x3FL), 1);
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)26, (short)(Long.rotateRight(ifFrequency, 8) & 0xFFL), 1);
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)27, (short)(ifFrequency & 0xFFL), 1);
    }

    public abstract void initTuner(boolean var1) throws UsbException;

    public String getUniqueID() {
        if (this.mDescriptor != null && this.mDescriptor.hasSerial()) {
            return this.mDescriptor.getSerial();
        }
        int serial = 0xFF & this.mDeviceDescriptor.iSerialNumber();
        return "SER#" + serial;
    }

    public abstract void setSampleRateFilters(int var1) throws DeviceException;

    public abstract TunerType getTunerType();

    public static TunerType identifyTunerType(Device device) throws DeviceException {
        DeviceHandle handle = new DeviceHandle();
        int reason = LibUsb.open(device, handle);
        if (reason != 0) {
            throw new DeviceException("couldn't open device - check permissions (udev.rule) [" + LibUsb.errorName(reason) + "]");
        }
        TunerType tunerClass = TunerType.UNKNOWN;
        try {
            RTL2832TunerController.claimInterface(handle);
            boolean resetRequired = false;
            try {
                RTL2832TunerController.writeRegister(handle, Block.USB, Address.USB_SYSCTL.getAddress(), 9, 1);
            }
            catch (LibUsbException e) {
                if (e.getErrorCode() < 0) {
                    Log.errorDialog("error performing dummy write - attempting device reset", e.getMessage());
                    resetRequired = true;
                }
                throw new DeviceException("error performing dummy write to device [" + LibUsb.errorName(e.getErrorCode()) + "] " + e.getMessage());
            }
            if (resetRequired) {
                reason = LibUsb.resetDevice(handle);
                try {
                    RTL2832TunerController.writeRegister(handle, Block.USB, Address.USB_SYSCTL.getAddress(), 9, 1);
                }
                catch (LibUsbException e2) {
                    Log.errorDialog("ERROR", "device reset attempted, but lost device handle.  Try restarting the application to use this device");
                    throw new DeviceException("couldn't reset device");
                }
            }
            RTL2832TunerController.initBaseband(handle);
            RTL2832TunerController.enableI2CRepeater(handle, true);
            boolean controlI2CRepeater = false;
            if (RTL2832TunerController.isTuner(TunerTypeCheck.E4K, handle, controlI2CRepeater)) {
                tunerClass = TunerType.ELONICS_E4000;
            } else if (RTL2832TunerController.isTuner(TunerTypeCheck.FC0013, handle, controlI2CRepeater)) {
                tunerClass = TunerType.FITIPOWER_FC0013;
            } else if (RTL2832TunerController.isTuner(TunerTypeCheck.R820T, handle, controlI2CRepeater)) {
                tunerClass = TunerType.RAFAELMICRO_R820T;
            } else if (RTL2832TunerController.isTuner(TunerTypeCheck.R828D, handle, controlI2CRepeater)) {
                tunerClass = TunerType.RAFAELMICRO_R828D;
            } else if (RTL2832TunerController.isTuner(TunerTypeCheck.FC2580, handle, controlI2CRepeater)) {
                tunerClass = TunerType.FCI_FC2580;
            } else if (RTL2832TunerController.isTuner(TunerTypeCheck.FC0012, handle, controlI2CRepeater)) {
                tunerClass = TunerType.FITIPOWER_FC0012;
            }
            RTL2832TunerController.enableI2CRepeater(handle, false);
            RTL2832TunerController.releaseInterface(handle);
            LibUsb.close(handle);
        }
        catch (Exception e) {
            Log.errorDialog("error while determining tuner type", e.getMessage());
        }
        return tunerClass;
    }

    public void release() {
        try {
            if (this.mBufferProcessor.isRunning()) {
                this.mBufferProcessor.stop();
            }
            LibUsb.releaseInterface(this.mDeviceHandle, 0);
            LibUsb.exit(null);
        }
        catch (Exception e) {
            Log.errorDialog("attempt to release USB interface failed", e.getMessage());
        }
    }

    public void resetUSBBuffer() throws LibUsbException {
        RTL2832TunerController.writeRegister(this.mDeviceHandle, Block.USB, Address.USB_EPA_CTL.getAddress(), 4098, 2);
        RTL2832TunerController.writeRegister(this.mDeviceHandle, Block.USB, Address.USB_EPA_CTL.getAddress(), 0, 2);
    }

    public static void initBaseband(DeviceHandle handle) throws LibUsbException {
        RTL2832TunerController.writeRegister(handle, Block.USB, Address.USB_SYSCTL.getAddress(), 9, 1);
        RTL2832TunerController.writeRegister(handle, Block.USB, Address.USB_EPA_MAXPKT.getAddress(), 2, 2);
        RTL2832TunerController.writeRegister(handle, Block.USB, Address.USB_EPA_CTL.getAddress(), 4098, 2);
        RTL2832TunerController.writeRegister(handle, Block.SYS, Address.DEMOD_CTL_1.getAddress(), 34, 1);
        RTL2832TunerController.writeRegister(handle, Block.SYS, Address.DEMOD_CTL.getAddress(), 232, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)1, 20, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)1, 16, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)21, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)22, 0, 2);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)22, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)23, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)24, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)25, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)26, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)27, 0, 1);
        int x = 0;
        while (x < sFIR_COEFFICIENTS.length) {
            RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)(28 + x), sFIR_COEFFICIENTS[x], 1);
            ++x;
        }
        RTL2832TunerController.writeDemodRegister(handle, Page.ZERO, (short)25, 5, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)147, 240, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)148, 15, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)17, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)4, 0, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ZERO, (short)97, 96, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ZERO, (short)6, 128, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ONE, (short)177, 27, 1);
        RTL2832TunerController.writeDemodRegister(handle, Page.ZERO, (short)13, 131, 1);
    }

    protected void deinitBaseband(DeviceHandle handle) throws IllegalArgumentException, UsbDisconnectedException, UsbException {
        RTL2832TunerController.writeRegister(handle, Block.SYS, Address.DEMOD_CTL.getAddress(), 32, 1);
    }

    protected static void setGPIOBit(DeviceHandle handle, byte bitMask, boolean enabled) throws LibUsbException {
        int value = RTL2832TunerController.readRegister(handle, Block.SYS, Address.GPO.getAddress(), 1);
        value = enabled ? (value |= bitMask) : (value &= ~bitMask);
        RTL2832TunerController.writeRegister(handle, Block.SYS, Address.GPO.getAddress(), value, 1);
    }

    protected static void setGPIOOutput(DeviceHandle handle, byte bitMask) throws LibUsbException {
        int value = RTL2832TunerController.readRegister(handle, Block.SYS, Address.GPD.getAddress(), 1);
        RTL2832TunerController.writeRegister(handle, Block.SYS, Address.GPO.getAddress(), value & ~bitMask, 1);
        value = RTL2832TunerController.readRegister(handle, Block.SYS, Address.GPOE.getAddress(), 1);
        RTL2832TunerController.writeRegister(handle, Block.SYS, Address.GPOE.getAddress(), value | bitMask, 1);
    }

    protected static void enableI2CRepeater(DeviceHandle handle, boolean enabled) throws LibUsbException {
        Page page = Page.ONE;
        short address = 1;
        int value = enabled ? 24 : 16;
        RTL2832TunerController.writeDemodRegister(handle, page, address, value, 1);
    }

    protected boolean isI2CRepeaterEnabled() throws DeviceException {
        int register = RTL2832TunerController.readDemodRegister(this.mDeviceHandle, Page.ONE, (short)1, 1);
        return register == 24;
    }

    protected static int readI2CRegister(DeviceHandle handle, byte i2CAddress, byte i2CRegister, boolean controlI2CRepeater) throws LibUsbException {
        short address = (short)(i2CAddress & 0xFF);
        ByteBuffer buffer = ByteBuffer.allocateDirect(1);
        buffer.put(i2CRegister);
        ((Buffer)buffer).rewind();
        ByteBuffer data = ByteBuffer.allocateDirect(1);
        if (controlI2CRepeater) {
            RTL2832TunerController.enableI2CRepeater(handle, true);
            RTL2832TunerController.write(handle, address, Block.I2C, buffer);
            RTL2832TunerController.read(handle, address, Block.I2C, data);
            RTL2832TunerController.enableI2CRepeater(handle, false);
        } else {
            RTL2832TunerController.write(handle, address, Block.I2C, buffer);
            RTL2832TunerController.read(handle, address, Block.I2C, data);
        }
        return data.get() & 0xFF;
    }

    protected void writeI2CRegister(DeviceHandle handle, byte i2CAddress, byte i2CRegister, byte value, boolean controlI2CRepeater) throws LibUsbException {
        short address = (short)(i2CAddress & 0xFF);
        ByteBuffer buffer = ByteBuffer.allocateDirect(2);
        buffer.put(i2CRegister);
        buffer.put(value);
        ((Buffer)buffer).rewind();
        if (controlI2CRepeater) {
            RTL2832TunerController.enableI2CRepeater(handle, true);
            RTL2832TunerController.write(handle, address, Block.I2C, buffer);
            RTL2832TunerController.enableI2CRepeater(handle, false);
        } else {
            RTL2832TunerController.write(handle, address, Block.I2C, buffer);
        }
    }

    protected static void writeDemodRegister(DeviceHandle handle, Page page, short address, int value, int length) throws LibUsbException {
        ByteBuffer buffer = ByteBuffer.allocateDirect(length);
        buffer.order(ByteOrder.BIG_ENDIAN);
        if (length == 1) {
            buffer.put((byte)(value & 0xFF));
        } else if (length == 2) {
            buffer.putShort((short)(value & 0xFFFF));
        } else {
            throw new IllegalArgumentException("Cannot write value greater than 16 bits to the register - length [" + length + "]");
        }
        short index = (short)(0x10 | page.getPage());
        short newAddress = (short)(address << 8 | 0x20);
        RTL2832TunerController.write(handle, newAddress, index, buffer);
        RTL2832TunerController.readDemodRegister(handle, Page.TEN, (short)1, length);
    }

    protected static int readDemodRegister(DeviceHandle handle, Page page, short address, int length) throws LibUsbException {
        byte index = page.getPage();
        short newAddress = (short)(address << 8 | 0x20);
        ByteBuffer buffer = ByteBuffer.allocateDirect(length);
        RTL2832TunerController.read(handle, newAddress, index, buffer);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        if (length == 2) {
            return buffer.getShort() & 0xFFFF;
        }
        return buffer.get() & 0xFF;
    }

    protected static void writeRegister(DeviceHandle handle, Block block, short address, int value, int length) throws LibUsbException {
        ByteBuffer buffer = ByteBuffer.allocateDirect(length);
        buffer.order(ByteOrder.BIG_ENDIAN);
        if (length == 1) {
            buffer.put((byte)(value & 0xFF));
        } else if (length == 2) {
            buffer.putShort((short)value);
        } else {
            throw new IllegalArgumentException("Cannot write value greater than 16 bits to the register - length [" + length + "]");
        }
        ((Buffer)buffer).rewind();
        RTL2832TunerController.write(handle, address, block, buffer);
    }

    protected static int readRegister(DeviceHandle handle, Block block, short address, int length) throws LibUsbException {
        ByteBuffer buffer = ByteBuffer.allocateDirect(2);
        RTL2832TunerController.read(handle, address, block, buffer);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        if (length == 2) {
            return buffer.getShort() & 0xFFFF;
        }
        return buffer.get() & 0xFF;
    }

    protected static void write(DeviceHandle handle, short address, Block block, ByteBuffer buffer) throws LibUsbException {
        RTL2832TunerController.write(handle, address, block.getWriteIndex(), buffer);
    }

    protected static void write(DeviceHandle handle, short value, short index, ByteBuffer buffer) throws LibUsbException {
        if (handle != null) {
            int transferred = LibUsb.controlTransfer(handle, (byte)64, (byte)0, value, index, buffer, 1000000L);
            if (transferred < 0) {
                throw new LibUsbException("error writing byte buffer", transferred);
            }
            if (transferred != buffer.capacity()) {
                throw new LibUsbException("transferred bytes [" + transferred + "] is not what was expected [" + buffer.capacity() + "]", transferred);
            }
        } else {
            throw new LibUsbException("device handle is null", -4);
        }
    }

    protected static void read(DeviceHandle handle, short address, short index, ByteBuffer buffer) throws LibUsbException {
        if (handle != null) {
            int transferred = LibUsb.controlTransfer(handle, (byte)-64, (byte)0, address, index, buffer, 1000000L);
            if (transferred < 0) {
                throw new LibUsbException("read error", transferred);
            }
            if (transferred != buffer.capacity()) {
                throw new LibUsbException("transferred bytes [" + transferred + "] is not what was expected [" + buffer.capacity() + "]", transferred);
            }
        } else {
            throw new LibUsbException("device handle is null", -4);
        }
    }

    protected static void read(DeviceHandle handle, short address, Block block, ByteBuffer buffer) throws LibUsbException {
        RTL2832TunerController.read(handle, address, block.getReadIndex(), buffer);
    }

    protected static boolean isTuner(TunerTypeCheck type, DeviceHandle handle, boolean controlI2CRepeater) {
        try {
            if (type == TunerTypeCheck.FC0012 || type == TunerTypeCheck.FC2580) {
                RTL2832TunerController.setGPIOOutput(handle, (byte)32);
                RTL2832TunerController.setGPIOBit(handle, (byte)32, true);
                RTL2832TunerController.setGPIOBit(handle, (byte)32, false);
            }
            int value = RTL2832TunerController.readI2CRegister(handle, type.getI2CAddress(), type.getCheckAddress(), controlI2CRepeater);
            if (type == TunerTypeCheck.FC2580) {
                return (value & 0x7F) == type.getCheckValue();
            }
            return value == type.getCheckValue();
        }
        catch (LibUsbException libUsbException) {
            return false;
        }
    }

    @Override
    public int getCurrentSampleRate() throws DeviceException {
        Log.println("RTL sampling at: " + this.mSampleRate.getRate());
        return this.mSampleRate.getRate();
    }

    public int getSampleRateFromTuner() throws DeviceException {
        try {
            int high = RTL2832TunerController.readDemodRegister(this.mDeviceHandle, Page.ONE, (short)159, 2);
            int low = RTL2832TunerController.readDemodRegister(this.mDeviceHandle, Page.ONE, (short)161, 2);
            int ratio = Integer.rotateLeft(high, 16) | low;
            int rate = this.mOscillatorFrequency * 0x400000 / ratio;
            SampleRate sampleRate = SampleRate.getClosest(rate);
            if (sampleRate.getRate() != rate) {
                this.setSampleRate(sampleRate);
                return sampleRate.getRate();
            }
        }
        catch (Exception e) {
            Log.println("RTL2832 Tuner Controller - cannot get current sample rate " + e.getMessage());
            throw e;
        }
        return DEFAULT_SAMPLE_RATE.getRate();
    }

    @Override
    public void setSampleRate(SampleRate sampleRate) throws DeviceException {
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)159, sampleRate.getRatioHighBits(), 2);
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)161, 0, 2);
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)1, 20, 1);
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)1, 16, 1);
        this.setSampleRateFilters(sampleRate.getRate());
        this.mSampleRate = sampleRate;
        if (this.mSampleRateMonitor != null) {
            this.mSampleRateMonitor.setSampleRate(this.mSampleRate.getRate());
        }
    }

    public void setSampleRateFrequencyCorrection(int ppm) throws DeviceException, UsbException {
        Log.println("Setting ppm to: " + ppm);
        int offset = -ppm * 4 * 0x400000 / 1000000;
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)63, offset & 0xFF, 1);
        RTL2832TunerController.writeDemodRegister(this.mDeviceHandle, Page.ONE, (short)62, Integer.rotateRight(offset, 8) & 0xFF, 1);
        double freq = Config.fcdFrequency;
        this.setFrequency((long)(freq * 1000.0));
    }

    public int getSampleRateFrequencyCorrection() throws UsbException {
        int high = RTL2832TunerController.readDemodRegister(this.mDeviceHandle, Page.ONE, (short)62, 1);
        int low = RTL2832TunerController.readDemodRegister(this.mDeviceHandle, Page.ONE, (short)63, 1);
        return Integer.rotateLeft(high, 8) | low;
    }

    public byte[] readEEPROM(DeviceHandle handle, short offset, int length) throws IllegalArgumentException {
        if (offset + length > 256) {
            throw new IllegalArgumentException("cannot read more than 256 bytes from EEPROM - requested to read to byte [" + (offset + length) + "]");
        }
        byte[] data = new byte[length];
        ByteBuffer buffer = ByteBuffer.allocateDirect(1);
        try {
            RTL2832TunerController.writeRegister(handle, Block.I2C, (short)-96, (byte)offset, 1);
        }
        catch (LibUsbException e) {
            Log.errorDialog("usb error while attempting to set read address to EEPROM register, prior to reading the EEPROM device descriptor", e.getMessage());
        }
        int x = 0;
        while (x < length) {
            try {
                RTL2832TunerController.read(handle, (short)-96, Block.I2C, buffer);
                data[x] = buffer.get();
                ((Buffer)buffer).rewind();
            }
            catch (Exception e) {
                Log.errorDialog("error while reading eeprom byte [" + x + "/" + length + "] aborting eeprom read and returning partially " + "filled descriptor byte array", e.getMessage());
                x = length;
            }
            ++x;
        }
        return data;
    }

    public void writeEEPROMByte(DeviceHandle handle, byte offset, byte value) throws IllegalArgumentException, UsbDisconnectedException, UsbException {
        if (offset < 0 || offset > 255) {
            throw new IllegalArgumentException("RTL2832 Tuner Controller - EEPROM offset must be within range of 0 - 255");
        }
        int offsetAndValue = Integer.rotateLeft(0xFF & offset, 8) | 0xFF & value;
        RTL2832TunerController.writeRegister(handle, Block.I2C, (short)-96, offsetAndValue, 2);
    }

    @Override
    public void setUsbSource(SourceUSB audioSource) {
        this.usbSource = audioSource;
        if (this.mBufferProcessor == null || !this.mBufferProcessor.isRunning()) {
            if (this.mBufferProcessor == null) {
                this.mBufferProcessor = new BufferProcessor(this.mThreadPoolManager);
            }
            if (!this.mBufferProcessor.isRunning()) {
                Thread thread = new Thread(this.mBufferProcessor);
                thread.setDaemon(true);
                thread.setName("RTL2832 Sample Processor");
                thread.start();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addListener(Listener<ComplexBuffer> listener) {
        Broadcaster<ComplexBuffer> broadcaster = this.mComplexBufferBroadcaster;
        synchronized (broadcaster) {
            this.mComplexBufferBroadcaster.addListener(listener);
            if (this.mBufferProcessor == null) {
                this.mBufferProcessor = new BufferProcessor(this.mThreadPoolManager);
            }
            if (!this.mBufferProcessor.isRunning()) {
                Thread thread = new Thread(this.mBufferProcessor);
                thread.setDaemon(true);
                thread.setName("RTL2832 Sample Processor");
                thread.start();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeListener(Listener<ComplexBuffer> listener) {
        Broadcaster<ComplexBuffer> broadcaster = this.mComplexBufferBroadcaster;
        synchronized (broadcaster) {
            this.mComplexBufferBroadcaster.removeListener(listener);
            if (!this.mComplexBufferBroadcaster.hasListeners()) {
                this.mBufferProcessor.stop();
            }
        }
    }

    public static String getTransferStatus(int status) {
        switch (status) {
            case 0: {
                return "TRANSFER COMPLETED (0)";
            }
            case 1: {
                return "TRANSFER ERROR (1)";
            }
            case 2: {
                return "TRANSFER TIMED OUT (2)";
            }
            case 3: {
                return "TRANSFER CANCELLED (3)";
            }
            case 4: {
                return "TRANSFER STALL (4)";
            }
            case 5: {
                return "TRANSFER NO DEVICE (5)";
            }
            case 6: {
                return "TRANSFER OVERFLOW (6)";
            }
        }
        return "UNKNOWN TRANSFER STATUS (" + status + ")";
    }

    @Override
    public DevicePanel getDevicePanel() throws IOException, DeviceException {
        return new RTLPanel();
    }

    @Override
    public void cleanup() throws IOException, DeviceException {
        if (this.mBufferProcessor != null && this.mBufferProcessor.isRunning()) {
            this.mBufferProcessor.stop();
        }
    }

    public static enum Address {
        USB_SYSCTL(8192),
        USB_CTRL(8208),
        USB_STAT(8212),
        USB_EPA_CFG(8516),
        USB_EPA_CTL(8520),
        USB_EPA_MAXPKT(8536),
        USB_EPA_MAXPKT_2(8538),
        USB_EPA_FIFO_CFG(8544),
        DEMOD_CTL(12288),
        GPO(12289),
        GPI(12290),
        GPOE(12291),
        GPD(12292),
        SYSINTE(12293),
        SYSINTS(12294),
        GP_CFG0(12295),
        GP_CFG1(12296),
        SYSINTE_1(12297),
        SYSINTS_1(12298),
        DEMOD_CTL_1(12299),
        IR_SUSPEND(12300);

        private int mAddress;

        private Address(int address) {
            this.mAddress = address;
        }

        public short getAddress() {
            return (short)this.mAddress;
        }
    }

    public static enum Block {
        DEMOD(0),
        USB(1),
        SYS(2),
        TUN(3),
        ROM(4),
        IR(5),
        I2C(6);

        private int mValue;

        private Block(int value) {
            this.mValue = value;
        }

        public int getValue() {
            return this.mValue;
        }

        public short getReadIndex() {
            return (short)Integer.rotateLeft(this.mValue, 8);
        }

        public short getWriteIndex() {
            return (short)(this.getReadIndex() | 0x10);
        }
    }

    public class BufferDispatcher
    implements Runnable {
        @Override
        public void run() {
            try {
                ArrayList buffers = new ArrayList();
                RTL2832TunerController.this.mFilledBuffers.drainTo(buffers);
                for (byte[] buffer : buffers) {
                    float[] samples = RTL2832TunerController.this.mSampleAdapter.convert(buffer);
                    RTL2832TunerController.this.mComplexBufferBroadcaster.broadcast(new ComplexBuffer(samples));
                    if (RTL2832TunerController.this.usbSource == null) continue;
                    RTL2832TunerController.this.usbSource.receive(samples);
                }
            }
            catch (Exception e) {
                Log.errorDialog("error duing rtl2832 buffer dispatcher run", e.getMessage());
            }
        }
    }

    public class BufferProcessor
    implements Runnable,
    TransferCallback {
        private ScheduledFuture<?> mSampleDispatcherTask;
        private LinkedTransferQueue<Transfer> mAvailableTransfers;
        private LinkedTransferQueue<Transfer> mTransfersInProgress = new LinkedTransferQueue();
        private AtomicBoolean mRunning = new AtomicBoolean();
        private ThreadPoolManager mThreadPoolManager;
        private ByteBuffer mLibUsbHandlerStatus;
        private boolean mCancel = false;

        public BufferProcessor(ThreadPoolManager manager) {
            this.mThreadPoolManager = manager;
        }

        @Override
        public void run() {
            if (this.mRunning.compareAndSet(false, true)) {
                try {
                    RTL2832TunerController.this.setSampleRate(RTL2832TunerController.this.mSampleRate);
                    RTL2832TunerController.this.resetUSBBuffer();
                }
                catch (DeviceException e) {
                    Log.errorDialog("couldn't start buffer processor", e.getMessage());
                    this.mRunning.set(false);
                }
                this.prepareTransfers();
                try {
                    this.mSampleDispatcherTask = this.mThreadPoolManager.scheduleFixedRate(ThreadPoolManager.ThreadType.SOURCE_SAMPLE_PROCESSING, new BufferDispatcher(), 20L, TimeUnit.MILLISECONDS);
                }
                catch (NullPointerException npe) {
                    Log.errorDialog("NPE! = tpm null: " + (this.mThreadPoolManager == null), npe.getMessage());
                }
                this.mLibUsbHandlerStatus = ByteBuffer.allocateDirect(4);
                ArrayList transfers = new ArrayList();
                this.mCancel = false;
                while (this.mRunning.get()) {
                    if (RTL2832TunerController.this.devicePulled) {
                        return;
                    }
                    this.mAvailableTransfers.drainTo(transfers);
                    for (Transfer transfer : transfers) {
                        int result = LibUsb.submitTransfer(transfer);
                        if (result == 0) {
                            this.mTransfersInProgress.add(transfer);
                            continue;
                        }
                        Log.errorDialog("ERROR", "Error submitting transfer [" + LibUsb.errorName(result) + "]");
                    }
                    int result = LibUsb.handleEventsTimeoutCompleted(null, 1000000L, this.mLibUsbHandlerStatus.asIntBuffer());
                    if (result != 0) {
                        Log.errorDialog("ERROR", "error handling events for libusb");
                    }
                    transfers.clear();
                    ((Buffer)this.mLibUsbHandlerStatus).rewind();
                }
                if (this.mCancel) {
                    for (Transfer transfer : this.mTransfersInProgress) {
                        LibUsb.cancelTransfer(transfer);
                    }
                    int result = LibUsb.handleEventsTimeoutCompleted(null, 1000000L, this.mLibUsbHandlerStatus.asIntBuffer());
                    if (result != 0) {
                        Log.errorDialog("ERROR", "error handling events for libusb during cancel");
                    }
                    ((Buffer)this.mLibUsbHandlerStatus).rewind();
                }
            }
        }

        public void stop() {
            this.mCancel = true;
            if (this.mRunning.compareAndSet(true, false) && this.mSampleDispatcherTask != null) {
                this.mThreadPoolManager.cancel(this.mSampleDispatcherTask);
                this.mSampleDispatcherTask = null;
                RTL2832TunerController.this.mFilledBuffers.clear();
            }
        }

        public boolean isRunning() {
            return this.mRunning.get();
        }

        @Override
        public void processTransfer(Transfer transfer) {
            if (RTL2832TunerController.this.devicePulled) {
                return;
            }
            this.mTransfersInProgress.remove(transfer);
            switch (transfer.status()) {
                case 0: 
                case 4: {
                    if (transfer.actualLength() <= 0) break;
                    ByteBuffer buffer = transfer.buffer();
                    byte[] data = new byte[transfer.actualLength()];
                    buffer.get(data);
                    ((Buffer)buffer).rewind();
                    if (!this.isRunning()) break;
                    RTL2832TunerController.this.mFilledBuffers.add(data);
                    break;
                }
                case 3: {
                    break;
                }
                default: {
                    Log.errorDialog("ERROR, perhaps device removed?", "USB data transfer error [" + RTL2832TunerController.getTransferStatus(transfer.status()) + "] transferred actual: " + transfer.actualLength() + "\nYou will probablly need to restart FoxTelem to reactivate this device.");
                    RTL2832TunerController.this.devicePulled = true;
                    return;
                }
            }
            this.mAvailableTransfers.add(transfer);
        }

        private void prepareTransfers() throws LibUsbException {
            if (this.mAvailableTransfers == null) {
                this.mAvailableTransfers = new LinkedTransferQueue();
                int x = 0;
                while (x < 16) {
                    Transfer transfer = LibUsb.allocTransfer();
                    if (transfer == null) {
                        throw new LibUsbException("couldn't allocate transfer", -11);
                    }
                    ByteBuffer buffer = ByteBuffer.allocateDirect(RTL2832TunerController.this.mBufferSize);
                    LibUsb.fillBulkTransfer(transfer, RTL2832TunerController.this.mDeviceHandle, (byte)-127, buffer, this, "Buffer", 1000000L);
                    this.mAvailableTransfers.add(transfer);
                    ++x;
                }
            }
        }
    }

    public class Descriptor {
        private byte[] mData;
        private ArrayList<String> mLabels = new ArrayList();

        public Descriptor(byte[] data) {
            if (data != null) {
                this.mData = data;
            } else {
                data = new byte[256];
            }
            this.getLabels();
        }

        public boolean isValid() {
            return this.mData[0] == 40 && this.mData[1] == 50;
        }

        public String getVendorID() {
            int id = Integer.rotateLeft(0xFF & this.mData[3], 8) | 0xFF & this.mData[2];
            return String.format("%04X", id);
        }

        public String getVendorLabel() {
            return this.mLabels.get(0);
        }

        public String getProductID() {
            int id = Integer.rotateLeft(0xFF & this.mData[5], 8) | 0xFF & this.mData[4];
            return String.format("%04X", id);
        }

        public String getProductLabel() {
            return this.mLabels.get(1);
        }

        public boolean hasSerial() {
            return this.mData[6] == -91;
        }

        public String getSerial() {
            return this.mLabels.get(2);
        }

        public boolean remoteWakeupEnabled() {
            int mask = 1;
            return (this.mData[7] & mask) == mask;
        }

        public boolean irEnabled() {
            int mask = 2;
            return (this.mData[7] & mask) == mask;
        }

        private void getLabels() {
            this.mLabels.clear();
            int start = 9;
            while (start < 256) {
                start = this.getLabel(start);
            }
        }

        private int getLabel(int start) {
            if (start > 254 || this.mData[start + 1] != 3) {
                return 256;
            }
            int length = 0xFF & this.mData[start];
            if (start + length > 255) {
                return 256;
            }
            byte[] data = Arrays.copyOfRange(this.mData, start + 2, start + length);
            String label = new String(data, Charset.forName("UTF-16LE"));
            this.mLabels.add(label);
            return start + length;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("RTL-2832 EEPROM Descriptor\n");
            sb.append("Vendor: ");
            sb.append(this.getVendorID());
            sb.append(" [");
            sb.append(this.getVendorLabel());
            sb.append("]\n");
            sb.append("Product: ");
            sb.append(this.getProductID());
            sb.append(" [");
            sb.append(this.getProductLabel());
            sb.append("]\n");
            sb.append("Serial: ");
            if (this.hasSerial()) {
                sb.append("yes [");
                sb.append(this.getSerial());
                sb.append("]\n");
            } else {
                sb.append("no\n");
            }
            sb.append("Remote Wakeup Enabled: ");
            sb.append(this.remoteWakeupEnabled() ? "yes" : "no");
            sb.append("\n");
            sb.append("IR Enabled: ");
            sb.append(this.irEnabled() ? "yes" : "no");
            sb.append("\n");
            if (this.mLabels.size() > 3) {
                sb.append("Additional Labels: ");
                int x = 3;
                while (x < this.mLabels.size()) {
                    sb.append(" [");
                    sb.append(this.mLabels.get(x));
                    sb.append("\n");
                    ++x;
                }
            }
            return sb.toString();
        }
    }

    public static enum Page {
        ZERO(0),
        ONE(1),
        TEN(10);

        private int mPage;

        private Page(int page) {
            this.mPage = page;
        }

        public byte getPage() {
            return (byte)(this.mPage & 0xFF);
        }
    }

    public static enum SampleMode {
        QUADRATURE,
        DIRECT;

    }

    public static enum SampleRate {
        RATE_0_240MHZ(7680, 240000, "0.240 MHz"),
        RATE_0_288MHZ(6400, 288000, "0.288 MHz"),
        RATE_0_960MHZ(1920, 960000, "0.960 MHz"),
        RATE_1_440MHZ(1280, 1440000, "1.440 MHz"),
        RATE_1_920MHZ(960, 1920000, "1.920 MHz"),
        RATE_2_304MHZ(800, 2304000, "2.304 MHz"),
        RATE_2_880MHZ(640, 2880000, "2.880 MHz");

        private int mRatioHigh;
        private int mRate;
        private String mLabel;

        private SampleRate(int ratioHigh, int rate, String label) {
            this.mRatioHigh = ratioHigh;
            this.mRate = rate;
            this.mLabel = label;
        }

        public int getRatioHighBits() {
            return this.mRatioHigh;
        }

        public int getRate() {
            return this.mRate;
        }

        public String getLabel() {
            return this.mLabel;
        }

        public String toString() {
            return this.mLabel;
        }

        public static SampleRate getClosest(int sampleRate) {
            SampleRate[] sampleRateArray = SampleRate.values();
            int n = sampleRateArray.length;
            int n2 = 0;
            while (n2 < n) {
                SampleRate rate = sampleRateArray[n2];
                if (rate.getRate() >= sampleRate) {
                    return rate;
                }
                ++n2;
            }
            return DEFAULT_SAMPLE_RATE;
        }
    }

    public class SampleRateMonitor
    implements Runnable {
        private static final int BUFFER_SIZE = 5;
        private int mTargetSampleRate;
        private int mSampleRateMinimum;
        private int mSampleRateMaximum;
        private int mNewTargetSampleRate;
        private FloatAveragingBuffer mRateErrorBuffer = new FloatAveragingBuffer(5);
        private AtomicBoolean mRateChanged = new AtomicBoolean();

        public SampleRateMonitor(int sampleRate) {
            this.setTargetRate(sampleRate);
        }

        public void setSampleRate(int sampleRate) {
            this.mNewTargetSampleRate = sampleRate;
            this.mRateChanged.set(true);
        }

        private void setTargetRate(int rate) {
            this.mTargetSampleRate = rate;
            this.mSampleRateMinimum = (int)((float)this.mTargetSampleRate * 0.95f);
            this.mSampleRateMaximum = (int)((float)this.mTargetSampleRate * 1.05f);
            int x = 0;
            while (x < 5) {
                this.mRateErrorBuffer.get(0.0f);
                ++x;
            }
        }

        @Override
        public void run() {
            if (this.mRateChanged.compareAndSet(true, false)) {
                this.setTargetRate(this.mNewTargetSampleRate);
                RTL2832TunerController.this.mSampleCounter.set(0);
                Log.println("monitor reset for new sample rate [" + this.mTargetSampleRate + "]");
            } else {
                int count = RTL2832TunerController.this.mSampleCounter.getAndSet(0);
                float current = (float)count / 20.0f;
                float average = (float)this.mSampleRateMinimum < current && current < (float)this.mSampleRateMaximum ? this.mRateErrorBuffer.get((float)this.mTargetSampleRate - current) : (float)this.mTargetSampleRate;
                StringBuilder sb = new StringBuilder();
                sb.append("[");
                if (RTL2832TunerController.this.mDescriptor != null) {
                    sb.append(RTL2832TunerController.this.mDescriptor.getSerial());
                } else {
                    sb.append("DESCRIPTOR IS NULL");
                }
                sb.append("] sample rate current [");
                sb.append(mDecimalFormatter.format(current));
                sb.append(" Hz ");
                sb.append(mPercentFormatter.format(100.0f * (current / (float)this.mTargetSampleRate)));
                sb.append("% ] error [");
                sb.append(average);
                sb.append(" Hz ] target ");
                sb.append(mDecimalFormatter.format(this.mTargetSampleRate));
                Log.println(sb.toString());
            }
        }
    }

    public static enum TunerTypeCheck {
        E4K(200, 2, 64),
        FC0012(198, 0, 161),
        FC0013(198, 0, 163),
        FC2580(172, 1, 86),
        R820T(52, 0, 105),
        R828D(116, 0, 105);

        private int mI2CAddress;
        private int mCheckAddress;
        private int mCheckValue;

        private TunerTypeCheck(int i2c, int address, int value) {
            this.mI2CAddress = i2c;
            this.mCheckAddress = address;
            this.mCheckValue = value;
        }

        public byte getI2CAddress() {
            return (byte)this.mI2CAddress;
        }

        public byte getCheckAddress() {
            return (byte)this.mCheckAddress;
        }

        public byte getCheckValue() {
            return (byte)this.mCheckValue;
        }
    }

    public class USBEventHandlingThread
    implements Runnable {
        private volatile boolean mAbort;

        public void abort() {
            this.mAbort = true;
        }

        @Override
        public void run() {
            while (!this.mAbort) {
                int result = LibUsb.handleEventsTimeout(null, 1000L);
                if (result == 0) continue;
                this.mAbort = true;
                Log.errorDialog("ERROR", "error handling usb events [" + LibUsb.errorName(result) + "]");
                throw new LibUsbException("Unable to handle USB events", result);
            }
        }
    }
}

